using System;
using System.Data;
using gov.va.med.vbecs.Common;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.ProlongedTransfusionReport;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stanislav Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/11/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Provides collection of input parameters and data retrieval for the prolonged transfusion time report.</summary>

	#endregion

	public class ProlongedTransfusionReport : VbecsReport
	{	
		private const int DelayedTransfusionTimeThresholdMinutes = 30; // BR_105.04

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5831"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null report object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5839"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Sort order is set to reverse chronological.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5840"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>No rules are broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5832"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Default constructor setting default parameters. 
		/// </summary>
		public ProlongedTransfusionReport()
			: base() 
		{
			SetInitialBrokenRules( BROKEN.FormClassName, BROKEN.RuleSets.BaseRuleSet );

			StartDate = EndDate = VBECSDateTime.GetDivisionCurrentDateTime().AddDays( -1 );

			// BR_105.1
			this.SortOrder = ProlongedTransfusionTimeReportSortOrder.ReverseChronological;

			this.IsDirty = false;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5835"> 
		///		<ExpectedInput>Valid start and end date.</ExpectedInput>
		///		<ExpectedOutput>
		///			Non-null result - DataTable; no exception is thrown. It is not feasible to predict how many 
		///			records will the report return without fully replicating it in unit tests. It is not also 
		///			practically possible to set up a unit test that will create set of data providing deterministic result.
		///		</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5836"> 
		///		<ExpectedInput>Start date is after end date.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates data for the prolonged transfusion time report.
		/// </summary>
		/// <returns><see cref="DataTable"/> containing data for the report.</returns>
		public virtual DataTable GetReportData()
		{
			VerifyObjectValidityBeforeGeneratingReport();

			return DAL.ProlongedTransfusionReport.GetReportData( this.StartDate, this.EndDate, DelayedTransfusionTimeThresholdMinutes, this.SortOrder );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5833"> 
		///		<ExpectedInput>Issue-to location sort order.</ExpectedInput>
		///		<ExpectedOutput>Issue-to location sort order.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5834"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets report sort order.
		/// </summary>
		public virtual Common.ProlongedTransfusionTimeReportSortOrder SortOrder
		{
			get
			{
				return (Common.ProlongedTransfusionTimeReportSortOrder)Convert.ToInt32(this.ParameterRow["SortOrder"]);
			}
			set
			{
				this.ParameterRow["SortOrder"] = value;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5891"> 
		///		<ExpectedInput>Valid start end and end dates, division code. </ExpectedInput>
		///		<ExpectedOutput>Non-negative integer.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5895"> 
		///		<ExpectedInput>Invalid date range (start date is after the end date).</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves record count estimate (the number is exact, but it can change 
		/// before <see cref="GetReportData"/> will be invoked) for the report.
		/// </summary>
		/// <returns>Record count estimate.</returns>
		public virtual int GetRecordCountEstimate()
		{
			VerifyObjectValidityBeforeGeneratingReport();

			return DAL.ProlongedTransfusionReport.GetReportCountEstimate( this.StartDate, this.EndDate, DelayedTransfusionTimeThresholdMinutes );
		}
		
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5893"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-null report criteria.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5894"> 
		///		<ExpectedInput>Invalid date range (start date is after the end date).</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Forms and returns report criteria string in a human-readable form.
		/// </summary>
		/// <returns>Report criteria string.</returns>
		public virtual string GetReportCriteriaText()
		{
			VerifyObjectValidityBeforeGeneratingReport();

			return String.Concat(
				GetDateRangeReportCriteria(),
				Environment.NewLine,
				StrRes.OtherMsg.UC105.SortOrderTitle().ResString,
				GetCurrentSortOrderName() );
		}


		/// <summary>
		/// Verifies that object is in valid state and throws exception if it's invalid.
		/// </summary>
		protected virtual void VerifyObjectValidityBeforeGeneratingReport()
		{
			//If its scheduled, by the time it prints, the print date might be in the past - which violates the rules
			//so we'll only do this check when not scheduling - we checked it at the time it was scheduled anyway
			if (this.IsScheduled)
				return;

			if( !this.IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToGenerateReportBecauseSomeRulesAreBroken( this.BrokenRulesCount, this.GetType().Name ).ResString ) );
		}

		/// <summary>
		/// Gets current sort order name.
		/// </summary>
		/// <returns>Current sort order name.</returns>
		protected virtual string GetCurrentSortOrderName()
		{
			switch( SortOrder )
			{
				case ProlongedTransfusionTimeReportSortOrder.IssueToLocation:
					return StrRes.OtherMsg.UC105.SortOrderNameIssueToLocation().ResString;
				default:
					return StrRes.OtherMsg.UC105.SortOrderNameReverseChronological().ResString;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6749"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6750"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates and prints report. Used by VbecsConsoleReport application.
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            DataTable _dtReport = this.GetReportData();
            _dtReport.TableName = "ProlongedTransfusion";
            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(this.GetReportCriteriaText());

            ReportDataSource reportSourceMain = new ReportDataSource();
            reportSourceMain.Name = "ProlongedTransfusionTimeDataSet";
            reportSourceMain.Value = _dtReport;

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ProlongedTransfusionTime.ProlongedTransfusionTimeReport.rdlc");

            LocalReport report = new LocalReport();
            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.DataSources.Add(reportSourceMain);
            report.LoadReportDefinition(stream);

            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
        }
	}
}
